% Copyright (c) 2014-2015, Serge Smirnoff (soundexpert.org)
% The code covered by FreeBSD License; see footer for full text.
% Version 2.2; see footer for ChangeLog.
% Matlab 2014b

% ----------------------------- Overview ----------------------------------

% The function compares two audio files (reference and output) piece-wise
% and computes Differense level (Df, dB) for each piece. Resulting vector
% of Df values is accompanied by diffrogram  an image showing this vector
% as a sequence of color coded bars. Color of each bar corresponds to Df
% value according to color map. Df values below -150dB are clipped to
% -150dB; Df value -Inf dB (perfect match of reference and output
% waveforms) is coded with Grey. Additionally each bar shows power spectral
% density estimate of corresponding piece of output signal. File name of
% returned diffrogram image includes min, median and max Df values of df
% vector (first and last values are not counted as they are often erroneous
% due to edge effects).
% 
% In most cases for correct calculation of Df levels the output signal must
% be time/phase aligned to the reference one beforehand. This operation of
% time-warping and phase-shifting is performed by diffrogram function using
% rectangular window WarpFrame. Resulting warped output signal is also
% returned by the function.

% -------------------------Syntax and description--------------------------

% DF = DIFFROGRAM(FREF, FOUT, WDIFF, 'OPTIONS') reads audio files named
% FREF and FOUT and returns three output arguments: the vector DF of Df
% values in dB, the diffrogram image and the warped output signal. The
% audio files can be of any sample rate and bit depth. However, accuracy of
% time warping depends on bandwidth of output signal and its sample rate.
% If sample rate of output signal is equal or less than sample rate of
% reference signal the former is up-sampled before warping.
% 
% WDIFF is the width of rectangular time window in milliseconds. It is used
% for calculating a single Df value in DF vector.
% 
% OPTIONS must include at least one of the followin arguments (case
% insensitive):
%
% - Left | Right | Mono | Stereo  mode of operation; in Stereo mode all
% operations are performed for Left and Right channels separately; in Mono
% mode both channels are combined together. Default [Stereo].
%
% - NoWarp  skip warping of output signal (if it is known as
% time/phase aligned already).
%
% - WarpFrame:integer  width of warping time frame in milliseconds;
% WarpFrame:0  warping in one piece; [30000]
%
% - WarpMargin:integer  number of samples of possible initial time
% misalignment between reference and output signals; the parameter can be
% useful mostly for pure sinusoidal signals and should be approx. equal to
% half of the period of reference signal (for example: 22 samples for 1 kHz
% at 44.1 kHz sample rate); for other signals it can be any integer greater
% than 0. The higher the value, the more the warping algorithm is prone to
% initial misalignment of output signal. For example, if output signal is a
% part of analog audio recording, it can be cut out with +/-WarpMargin
% accuracy. Though, with increase of  WarpMargin the computation time
% increases as well. [20]
%
% - NoLowpass  skip low-pass filtering of output signal before warping (if
% this operation has been already performed outside).
%
% - ColorMap:integer  return of color map only of the size specified in
% pixels; all other input arguments are ignored. [300]
% 
% 
% DIFFROGRAM(FREF, FOUT, WDIFF, 'OPTIONS') without output arguments returns
% diffrogram image and warped output signal (if warping was applied).

% ------------------------- Examples of usage -----------------------------

% diffrogram('ref.wav', 'out.wav', 100, 'Mono NoWarp')
% diffrogram('ref.wav', 'out.wav', 400, 'Mono NoLowpass WarpMargin:1')
% df = diffrogram('ref.flac', 'out.wav', 400, 'Mono WarpFrame:400');
%
% See "diffrogram_examples.pdf" for details. See also the article
% "Diffrogram: visualization of signal differences in audio research" -
% http://soundexpert.org/news/-/blogs/visualization-of-distortion

% -------------------------------------------------------------------------


function df = diffrogram(fref, fout, Wdiff, options)
disp('**********************************************')
disp('***            Diffrogram V2.2             ***')
disp('***            soundexpert.org             ***')
disp('**********************************************')

options = lower(options);

% Return of ColorMap only
if ~isempty(strfind(options, 'colormap'))
    si = textscan(options, '%s');
    tf = strncmp('colormap', si{:}, 8);
    cc = char(si{:}(tf));
    NNmap = str2double(cc(10:end));
    if isnan(NNmap), NNmap = 300; end % default value
    disp(['Color map output only, ' num2str(NNmap+1) 'x' num2str(NNmap) 'px'])
    colorM = cmap(NNmap);
    imwrite(colorM, 'diffrogram-color-map.png', 'bitdepth',16)
    df = NaN;
    return
end

[ref,Fsr] = audioread(fref);
[out,Fso] = audioread(fout);

% Define window Wwarp for piece-wise warping (ms)
if ~isempty(strfind(options, 'warpframe:'))
    si = textscan(options, '%s');
    tf = strncmp('warpframe:', si{:}, 10);
    cc = char(si{:}(tf));
    Wwarp = str2double(cc(11:end));
else
    Wwarp = 30000; % default value
end

Wwarp = round(Wwarp * Fsr / 1000);
Wdiff = round(Wdiff * Fsr / 1000);

% Define WarpMargin value
if ~isempty(strfind(options, 'warpmargin:'))
    si = textscan(options, '%s');
    tf = strncmp('warpmargin:', si{:}, 11);
    cc = char(si{:}(tf));
    WarpMargin = str2double(cc(12:end));
    WarpMargin = abs(round(WarpMargin));
    if WarpMargin < 1, WarpMargin = 1; end
else
    WarpMargin = 20; % default value
end

% Define mode of operation: Left|Right|Mono|Stereo
CHref = min(size(ref));
CHout = min(size(out));
if ~isempty(strfind(options, 'left'))
    if CHref == 1 && CHout == 2
        out = out(:,1);
    elseif CHref == 2 && CHout == 1
        ref = ref(:,1);
    elseif CHref == 1 && CHout == 1
        error('Either REF or OUT must have 2 channels for LEFT mode')
    elseif CHref == 2 && CHout == 2
        ref = ref(:,1);
        out = out(:,1);
    else
        error('More than 2 channels are not supported')
    end
    [ref,out] = NormByRef1(ref,out);
    Channel = 'left';
elseif ~isempty(strfind(options, 'right'))
    if CHref == 1 && CHout == 2
        out = out(:,2);
    elseif CHref == 2 && CHout == 1
        ref = ref(:,2);
    elseif CHref == 1 && CHout == 1
        error('Either REF or OUT must have 2 channels for RIGHT mode')
    elseif CHref == 2 && CHout == 2
        ref = ref(:,2);
        out = out(:,2);
    else
        error('More than 2 channels are not supported')
    end
    [ref,out] = NormByRef1(ref,out);
    Channel = 'right';
elseif ~isempty(strfind(options, 'mono'))
    if CHref == 1 && CHout == 2
        error('REF also must have 2 channels for 2ch OUT and MONO mode')
    elseif CHref == 2 && CHout == 1
        ref = (ref(:,1) + ref(:,2)) ./ 2;
        [ref,out] = NormByRef1(ref,out);
    elseif CHref == 1 && CHout == 1
        [ref,out] = NormByRef1(ref,out);
    elseif CHref == 2 && CHout == 2
        [ref(:,1),out(:,1)] = NormByRef1(ref(:,1),out(:,1));
        [ref(:,2),out(:,2)] = NormByRef1(ref(:,2),out(:,2));
        ref = (ref(:,1) + ref(:,2)) ./ 2;
        out = out(:,1) + out(:,2);
        [ref,out] = NormByRef1(ref,out);
    else
        error('More than 2 channels are not supported')
    end
    Channel = 'mono';
else
    if CHref == 1 && CHout == 2
        error('Both REF and OUT must have 2 channels for STEREO mode')
    elseif CHref == 2 && CHout == 1
        error('Both REF and OUT must have 2 channels for STEREO mode')
    elseif CHref == 1 && CHout == 1
        error('Both REF and OUT must have 2 channels for STEREO mode')
    elseif CHref == 2 && CHout == 2
        [ref(:,1),out(:,1)] = NormByRef1(ref(:,1),out(:,1));
        [ref(:,2),out(:,2)] = NormByRef1(ref(:,2),out(:,2));
    else
        error('More than 2 channels are not supported')
    end
    Channel = 'stereo';
end

disp(['FileRef: ' fref ', ' num2str(CHref) ' channel(s), ' num2str(Fsr) ' Hz'])
disp(['FileOut: ' fout ', ' num2str(CHout) ' channel(s), ' num2str(Fso) ' Hz'])
disp(['Df window: ' num2str(Wdiff*1000/Fsr) ' ms'])
disp('---- Options ----')
disp(['Mode: ' Channel])

% Upsample output audio if necessary
if Fso <= Fsr && isempty(strfind(options, 'nowarp'))
    P = ceil(4*Fsr/Fso);
    Q = 1;
    disp(['Upsampling of OUT: x' num2str(P)])
    out = resample(out,P,Q,1000);
else
    disp('Upsampling of OUT: no')
end

% LowPass output audio if necessary
if Fso > Fsr && isempty(strfind(options, 'nolowpass'))
    disp('Low-pass filtering of OUT: yes')
    b = fir1(4096,Fsr/Fso);
    out = filtfilt(b,1,out);
else
    disp('Low-pass filtering of OUT: no')
end

% Warp output audio if necessary
if isempty(strfind(options, 'nowarp'))
    Lref = length(ref);
    if Wwarp == 0 || Lref <= Wwarp
        disp(['Warping: in one piece (' num2str(round(Lref*1000/Fsr)) ' ms)'])
    else
        disp(['Warping: frame-wise (' num2str(round(Wwarp*1000/Fsr)) ' ms)'])
    end
    disp(['WarpMargin: ' num2str(WarpMargin) ' sample(s)'])
    [warp,ddf] = WarpAsRef(ref, out, Wwarp, WarpMargin);
    wavname = [fout '(' num2str(floor(Fso/1000)) ')_' ...
        'warp_' Channel '_' num2str(round(Wwarp*1000/Fsr)) '.wav'];
    audiowrite(wavname, warp, Fsr, 'BitsPerSample',32);
    disp('Df values for warp frames:');
    disp(num2str(ddf,'%8.4f'));
else
    if length(out) ~= length(ref)
        error('Lengths of REF and OUT must match for NoWarp mode')
    end
    warp = out;
    disp('Warping: no')
end

% Compute df and create diffrogram
if strcmpi(Channel,'left') ...
        || strcmpi(Channel,'right') ...
        || strcmpi(Channel,'mono')
    [df, sp] = dfsp1(ref, warp, Wdiff, Fsr);
    % sp(:) = -75; % without spectrogram
    % df(:) = -Inf; % Grey diffrogram
    [imdf, Min_df, Med_df, Max_df] = dfsp2img1(df, sp);
    imgname = [fout '(' num2str(floor(Fso/1000)) ')_' ...
        fref '(' num2str(floor(Fsr/1000)) ')_' ...
        Channel '_' num2str(Wdiff*1000/Fsr) ...
        num2str(Min_df,'%+8.4f') ...
        num2str(Med_df,'%+8.4f') ...
        num2str(Max_df,'%+8.4f') '.png'];
    imwrite(imdf, imgname, 'bitdepth',16)
    
elseif strcmpi(Channel,'stereo')
    [df1, sp] = dfsp1(ref(:,1), warp(:,1), Wdiff, Fsr);
    [imdf, Min_df, Med_df, Max_df] = dfsp2img1(df1, sp);
    imgname = [fout '(' num2str(floor(Fso/1000)) ')_' ...
        fref '(' num2str(floor(Fsr/1000)) ')_' ...
        'left_' num2str(Wdiff*1000/Fsr) ...
        num2str(Min_df,'%+8.4f') ...
        num2str(Med_df,'%+8.4f') ...
        num2str(Max_df,'%+8.4f') '.png'];
    imwrite(imdf, imgname, 'bitdepth',16)
    
    [df2, sp] = dfsp1(ref(:,2), warp(:,2), Wdiff, Fsr);
    [imdf, Min_df, Med_df, Max_df] = dfsp2img1(df2, sp);
    imgname = [fout '(' num2str(floor(Fso/1000)) ')_' ...
        fref '(' num2str(floor(Fsr/1000)) ')_' ...
        'right_' num2str(Wdiff*1000/Fsr) ...
        num2str(Min_df,'%+8.4f') ...
        num2str(Med_df,'%+8.4f') ...
        num2str(Max_df,'%+8.4f') '.png'];
    imwrite(imdf, imgname, 'bitdepth',16)
    
    [df3, sp] = dfsp1((ref(:,1)+ref(:,2))./2, (warp(:,1)+warp(:,2))./2, Wdiff, Fsr);
    [imdf, Min_df, Med_df, Max_df] = dfsp2img1(df3, sp);
    imgname = [fout '(' num2str(floor(Fso/1000)) ')_' ...
        fref '(' num2str(floor(Fsr/1000)) ')_' ...
        'stereo_' num2str(Wdiff*1000/Fsr) ...
        num2str(Min_df,'%+8.4f') ...
        num2str(Med_df,'%+8.4f') ...
        num2str(Max_df,'%+8.4f') '.png'];
    imwrite(imdf, imgname, 'bitdepth',16)
    
    df = [df1, df2, df3];
end

disp('**********************************************')

return

% ------------------------- sub-functions ---------------------------------

function [warp,ddf] = WarpAsRef(ref, out, Wwarp, WarpMargin)

Lref = length(ref);
Lout = length(out);
SCL = Lout/Lref;
MarginOut = WarpMargin * SCL;

if Wwarp == 0 || Lref <= Wwarp
    % Search for local minimum of Df in one piece
    [st,Df] = fminsearch(@(st) DfWarp(ref,out,st,WarpMargin),[SCL,0], ...
        optimset('TolX',1e-5, 'TolFun',1e-5, ...
        'Display','iter', 'MaxIter',500, 'MaxFunEvals',1000));
    if Df==-9999, Df = -Inf; end
    trw = 1:Lout;
    trw_u = trw(1)+st(2) : st(1) : trw(end)+st(2);
    warp = interp1(trw, out, trw_u', 'spline', 0);
    [~,warp] = CutByRef(ref,warp,WarpMargin);
    ddf = Df;
    disp(['Df: ' num2str(Df,'%8.4f') ...
        '  SCL: ' num2str(st(1),'%10.6f') ...
        '  TAU: ' num2str(st(2),'%+8.4f')]);
    return
end

N = floor(Lref / Wwarp);
warp = zeros(size(ref));
ddf = zeros(N,1);

if Lref > N * Wwarp, A = 1; else A = 0; end
disp(['Number of frames: ' num2str(N+A)])
% Search for local minimum of Df piece-wise
RE = 0;
for ii=1:N
    RB = RE + 1;
    RE = RB + Wwarp - 1;
    OB = RB*SCL-MarginOut; if OB < 1 || RB == 1, OB=1; end
    OE = RE*SCL+MarginOut; if OE > Lout, OE = Lout; end
    samRef = ref(RB:RE,:);
    samOut = out(round(OB):round(OE),:);
    [st,Df] = fminsearch(@(st) DfWarp(samRef,samOut,st,WarpMargin),[SCL,0], ...
        optimset('TolX',1e-5, 'TolFun',1e-5, ...
        'Display','iter', 'MaxIter',500, 'MaxFunEvals',1000));
    if Df==-9999, Df = -Inf; end
    trw = 1:length(samOut);
    trw_u = trw(1)+st(2) : st(1) : trw(end)+st(2);
    samWarp = interp1(trw, samOut, trw_u', 'spline', 0);
    [~,warp(RB:RE,:)] = CutByRef(samRef,samWarp,WarpMargin);
    ddf(ii) = Df;
    disp(['Piece: ' num2str(ii) '/' num2str(N+A) ...
        '  Df: ' num2str(Df,'%8.4f') ...
        '  SCL: ' num2str(st(1),'%10.6f') ...
        '  TAU: ' num2str(st(2),'%+8.4f')]);
end
% ... last piece
if Lref>RE
    Fn = Lref - RE;
    RB = Lref - Wwarp + 1;
    RE = Lref;
    OB = RB*SCL-MarginOut; if OB < 1, OB = 1; end
    OE = RE*SCL+MarginOut; if OE > Lout, OE = Lout; end
    samRef = ref(RB:RE,:);
    samOut = out(round(OB):round(OE),:);
    [st,Df] = fminsearch(@(st) DfWarp(samRef,samOut,st,WarpMargin),[SCL,0], ...
        optimset('TolX',1e-5, 'TolFun',1e-5, ...
        'Display','iter', 'MaxIter',500, 'MaxFunEvals',1000));
    if Df==-9999, Df = -Inf; end
    trw = 1:length(samOut);
    trw_u = trw(1)+st(2) : st(1) : trw(end)+st(2);
    samWarp = interp1(trw, samOut, trw_u', 'spline', 0);
    [~,samWarp] = CutByRef(samRef,samWarp,WarpMargin);
    warp(end-Fn+1:end,:) = samWarp(end-Fn+1:end,:);
    ddf = [ddf; Df];
    disp(['Piece: ' num2str(ii+1) '/' num2str(N+A) ...
        '  Df: ' num2str(Df,'%8.4f') ...
        '  SCL: ' num2str(st(1),'%10.6f') ...
        '  TAU: ' num2str(st(2),'%+8.4f')]);
end

return

function Df = DfWarp(ref, out, st, WarpMargin)
trw = 1:length(out);
trw_u = trw(1)+st(2) : st(1) : trw(end)+st(2);
warp = interp1(trw, out, trw_u', 'spline', 0);
ref = ref(1+WarpMargin : end-WarpMargin, :);
[~,warp] = CutByRef(ref,warp,WarpMargin);
[~,~,Df] = DfLRS(ref,warp);
return

function [ref,out] = CutByRef(ref, out, Lag)
Dlen = length(ref)-length(out);
DL = abs(Dlen);
HDB = floor(DL/2);
HDE= DL - HDB;

S = min(size(ref));

if Dlen < 0
    ref = [zeros(HDB,S); ref; zeros(HDE,S)];
elseif 	Dlen > 0
    out = [zeros(HDB,S); out; zeros(HDE,S)];
end

switch S
    case 1
        [~,Maxind] = max(xcorr(ref,out,Lag));
    case 2
        [~,Maxind] = max(xcorr(ref(:,1)+ref(:,2),out(:,1)+out(:,2),Lag));
end

if Maxind < (Lag+1)
    Shift = (Lag+1) - Maxind;
    out = [out(Shift+1:end,:);zeros(Shift,S)];
elseif Maxind > (Lag+1)
    Shift = Maxind - (Lag+1);
    out = [zeros(Shift,S);out(1:end-Shift,:)];
end

if Dlen < 0
    ref = ref(1+HDB:end-HDE,:);
    out = out(1+HDB:end-HDE,:);
end
return

function [DfL,DfR,DfS] = DfLRS(ref,out)
S = min(size(ref));
switch S
    case 1
        
        r = corrcoef(ref,out);
        if ~isnan(r(1,1)) && ~isnan(r(1,2)) && ~isnan(r(2,1)) && ~isnan(r(2,2))
            %both "ref" and "out" are NOT constant
            DfS = 10*log10(1-r(1,2));
        elseif isnan(r(1,1)) && isnan(r(1,2)) && isnan(r(2,1)) && isnan(r(2,2))
            %both "ref" and "out" are constant
            DfS = -Inf;
        else
            %only "ref" or only "out" is constant
            DfS = 0;
        end
        DfL = NaN;
        DfR = NaN;
        
    case 2
        
        r = corrcoef(ref(:,1),out(:,1));
        if ~isnan(r(1,1)) && ~isnan(r(1,2)) && ~isnan(r(2,1)) && ~isnan(r(2,2))
            %both "ref" and "out" are NOT constant
            DfL = 10*log10(1-r(1,2));
        elseif isnan(r(1,1)) && isnan(r(1,2)) && isnan(r(2,1)) && isnan(r(2,2))
            %both "ref" and "out" are constant
            DfL = -Inf;
        else
            %only "ref" or only "out" is constant
            DfL = 0;
        end
        
        r = corrcoef(ref(:,2),out(:,2));
        if ~isnan(r(1,1)) && ~isnan(r(1,2)) && ~isnan(r(2,1)) && ~isnan(r(2,2))
            %both "ref" and "out" are NOT constant
            DfR = 10*log10(1-r(1,2));
        elseif isnan(r(1,1)) && isnan(r(1,2)) && isnan(r(2,1)) && isnan(r(2,2))
            %both "ref" and "out" are constant
            DfR = -Inf;
        else
            %only "ref" or only "out" is constant
            DfR = 0;
        end
        
        [ref(:,1),out(:,1)] = NormByRef1(ref(:,1),out(:,1));
        [ref(:,2),out(:,2)] = NormByRef1(ref(:,2),out(:,2));
        r = corrcoef(ref(:,1)+ref(:,2),out(:,1)+out(:,2));
        if ~isnan(r(1,1)) && ~isnan(r(1,2)) && ~isnan(r(2,1)) && ~isnan(r(2,2))
            %both "ref" and "out" are NOT constant
            DfS = 10*log10(1-r(1,2));
        elseif isnan(r(1,1)) && isnan(r(1,2)) && isnan(r(2,1)) && isnan(r(2,2))
            %both "ref" and "out" are constant
            DfS = -Inf;
        else
            %only "ref" or only "out" is constant
            DfS = 0;
        end
        
end
if isinf(DfS), DfS = -9999; end
if isinf(DfL), DfL = -9999; end
if isinf(DfR), DfR = -9999; end
return

function [df, sp] = dfsp1(ref, out, Wdiff, Fsr)

Fmin = 20; % [Hz]
Fmax = Fsr/2; % [Hz]
Fz = 24000; % [Hz] reference frequency for spectrogram height (100px)
FftMin = 100000; % [samples]

if FftMin < Wdiff, FftMin = Wdiff; end
H = round(100 * (log10(Fmax)-log10(Fmin)) / (log10(Fz)-log10(Fmin)));
wLog = logspace(log10(pi*(Fmin/(Fsr/2))),log10(pi*(Fmax/(Fsr/2))),H);

N = floor(length(ref)/Wdiff);
df = zeros(N,1);
sp = zeros(N,H);

Kscale = 0.1414224; % scale factor for spectrogram (1kHz,48000,0dBFS)
E = 0;
for ii = 1:N
    
    B = E + 1;
    E = B + Wdiff - 1;
    samRef = ref(B:E,:);
    samOut = out(B:E,:);
    
    r = corrcoef(samRef,samOut);
    if ~isnan(r(1,1)) && ~isnan(r(1,2)) && ~isnan(r(2,1)) && ~isnan(r(2,2))
        % both REF and OUT are not constant
        df(ii) = 10*log10(1-r(1,2));
    elseif isnan(r(1,1)) && isnan(r(1,2)) && isnan(r(2,1)) && isnan(r(2,2))
        % both REF and OUT are constant
        df(ii) = -Inf;
    else
        % only REF or only OUT is constant
        df(ii) = 0;
    end
    
    [sps,w] = periodogram(samOut,[],FftMin);
    
    sp(ii,1) = mean(sps(w<=wLog(1)));
    for jj = 2:H
        sp(ii,jj) = mean(sps(w>wLog(jj-1) & w<=wLog(jj)));
    end
    
    Pout = sqrt(mean(samOut .^ 2));
    Psp = sqrt(mean(sp(ii,:) .^ 2));
    if Psp~=0
        Kp = (Pout / Psp) * Kscale;
        sp(ii,:) = sp(ii,:) .* Kp;
    end
    
    sp(ii,:) = 10 .* log10(sp(ii,:));
end

return

function [imdf, Min_df, Med_df, Max_df] = dfsp2img1(df, sp)
% Create image (png16) of diffrogram
MinDf = -150; % dB
MinPw = -150; % dB

Len = length(df);
Wdth = size(sp,2);
Nmap = 1500;

df(df<MinDf & ~isinf(df)) = MinDf;
Min_df = min(df(2:end-1));% reject the first and the last Df values
Med_df = median(df(2:end-1));% reject the first and the last Df values
Max_df = max(df(2:end-1));% reject the first and the last Df values
df(df>0) = 0;
df = round(Nmap + df .* ((Nmap-1)/Nmap*10)) + 1;
df(isinf(df)) = 1;

sp(sp<MinPw) = MinPw;
sp(sp>0) = 0;
sp = round(1 - sp .* ((Nmap-1)/Nmap*10));

colorM = cmap(Nmap);

imdf = zeros(Wdth,Len,3);
for ii = 1:Len
    for jj = 1:Wdth
        imdf(Wdth-jj+1,ii,:) = colorM(sp(ii,jj),df(ii),:);
    end
end

return

function colorM = cmap(Nmap)
% Build ColorMap with Grey scale for -Inf [dB]
m = [0.718, 0.000, 0.718];
b = [0.316, 0.316, 0.991];
c = [0.000, 0.559, 0.559];
g = [0.000, 0.592, 0.000];
y = [0.527, 0.527, 0.000];
r = [0.847, 0.057, 0.057];

gamma = 2;
Nseg = round(Nmap/5);

lin = linspace(0,1,Nseg);
mb = zeros(Nseg,3);
mb(:,1) = ((1-lin).*m(1)^gamma + lin.*b(1)^gamma).^(1/gamma);
mb(:,2) = ((1-lin).*m(2)^gamma + lin.*b(2)^gamma).^(1/gamma);
mb(:,3) = ((1-lin).*m(3)^gamma + lin.*b(3)^gamma).^(1/gamma);

lin = linspace(0,1,Nseg+1);
bc = zeros(Nseg+1,3);
bc(:,1) = ((1-lin).*b(1)^gamma + lin.*c(1)^gamma).^(1/gamma);
bc(:,2) = ((1-lin).*b(2)^gamma + lin.*c(2)^gamma).^(1/gamma);
bc(:,3) = ((1-lin).*b(3)^gamma + lin.*c(3)^gamma).^(1/gamma);

cg = zeros(Nseg+1,3);
cg(:,1) = ((1-lin).*c(1)^gamma + lin.*g(1)^gamma).^(1/gamma);
cg(:,2) = ((1-lin).*c(2)^gamma + lin.*g(2)^gamma).^(1/gamma);
cg(:,3) = ((1-lin).*c(3)^gamma + lin.*g(3)^gamma).^(1/gamma);

gy = zeros(Nseg+1,3);
gy(:,1) = ((1-lin).*g(1)^gamma + lin.*y(1)^gamma).^(1/gamma);
gy(:,2) = ((1-lin).*g(2)^gamma + lin.*y(2)^gamma).^(1/gamma);
gy(:,3) = ((1-lin).*g(3)^gamma + lin.*y(3)^gamma).^(1/gamma);

yr = zeros(Nseg+1,3);
yr(:,1) = ((1-lin).*y(1)^gamma + lin.*r(1)^gamma).^(1/gamma);
yr(:,2) = ((1-lin).*y(2)^gamma + lin.*r(2)^gamma).^(1/gamma);
yr(:,3) = ((1-lin).*y(3)^gamma + lin.*r(3)^gamma).^(1/gamma);

colorV = [mb; bc(2:end,:); cg(2:end,:); gy(2:end,:); yr(2:end,:)];

colorR = smooth(colorV(:,1),Nseg);
colorG = smooth(colorV(:,2),Nseg);
colorB = smooth(colorV(:,3),Nseg);

colorV = [colorR colorG colorB];

Klum = mean(colorV(:));
Ndark = round(Nmap*Klum);
Nlight = Nmap - Ndark + 1;

colorM = zeros(Nmap, Nmap+1, 3);
colorM(1:Nmap,1,:) = interp1([1;Nmap],[1 1 1; 0 0 0],1:Nmap);
for ii = 1:Nmap
    colorM(1:Nlight,ii+1,:) = interp1([1;Nlight],[1 1 1; colorV(ii,:)],1:Nlight);
    colorM(Nlight:end,ii+1,:) = interp1([1;Ndark],[colorV(ii,:); 0 0 0],1:Ndark);
end

return

function [ref,out] = NormByRef1(ref,out)

ref = detrend(ref,'constant');
out = detrend(out,'constant');

Px = sqrt(mean(ref.^2));
Py = sqrt(mean(out.^2));
Kp = Px / Py;
out = out .* Kp;

return


% ---------------------------- Changelog ----------------------------------

% V2.2
% - bugfix with lower case for "ColorMap" option
% - direct colormap output without df and sp vectors
% - added -Inf column (greyscale) to color map
% - first public release
 
% ------------------------- FreeBSD License -------------------------------

% Copyright (c) 2014-2015, Serge Smirnoff (soundexpert.org)
% All rights reserved.
% 
% Redistribution and use in source and binary forms, with or without
% modification, are permitted provided that the following conditions are met:
% 
% 1. Redistributions of source code must retain the above copyright notice, this
%    list of conditions and the following disclaimer.
% 2. Redistributions in binary form must reproduce the above copyright notice,
%    this list of conditions and the following disclaimer in the documentation
%    and/or other materials provided with the distribution.
% 
% THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
% ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
% WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
% DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
% ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
% (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
% LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
% ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
% (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
% SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
% 
% The views and conclusions contained in the software and documentation are those
% of the authors and should not be interpreted as representing official policies,
% either expressed or implied, of the FreeBSD Project.